<?php
declare(strict_types=1);

namespace PrinsFrank\Standards\InternationalCallPrefix;

use PrinsFrank\Standards\Country\CountryAlpha2;
use PrinsFrank\Standards\Country\CountryAlpha3;
use PrinsFrank\Standards\Country\CountryName;
use PrinsFrank\Standards\Country\CountryNumeric;

/**
 * @standard ITU-T E.164
 * @source https://www.itu.int/dms_pub/itu-t/opb/sp/T-SP-E.164C-2011-PDF-E.pdf
 *
 * @manually-updated
 */
enum InternationalCallPrefix: string {
    case _0 = '0';
    case _00 = '00';
    case _000 = '000';
    case _001 = '001';
    case _0011 = '0011';
    case _009 = '009';
    case _00x = '00x';
    case _010 = '010';
    case _011 = '011';
    case _1xx0 = '1xx0';
    case _8_10 = '8~10';

    /** @deprecated Used to be used by Cuba, which now uses 00 */
    case _119 = '119';

    public static function forCountry(CountryAlpha2|CountryAlpha3|CountryName|CountryNumeric $country): self {
        if ($country instanceof CountryAlpha2 === false) {
            $country = $country->toCountryAlpha2();
        }

        return match($country) {
            CountryAlpha2::Samoa => self::_0,
            CountryAlpha2::Afghanistan,
            CountryAlpha2::Aland_Islands,
            CountryAlpha2::Albania,
            CountryAlpha2::Algeria,
            CountryAlpha2::Andorra,
            CountryAlpha2::Angola,
            CountryAlpha2::Antarctica,
            CountryAlpha2::Argentina,
            CountryAlpha2::Armenia,
            CountryAlpha2::Aruba,
            CountryAlpha2::Austria,
            CountryAlpha2::Azerbaijan,
            CountryAlpha2::Bahrain,
            CountryAlpha2::Bangladesh,
            CountryAlpha2::Belgium,
            CountryAlpha2::Belize,
            CountryAlpha2::Benin,
            CountryAlpha2::Bhutan,
            CountryAlpha2::Bolivia,
            CountryAlpha2::Bonaire_Sint_Eustatius_and_Saba,
            CountryAlpha2::Bosnia_and_Herzegovina,
            CountryAlpha2::Botswana,
            CountryAlpha2::Bouvet_Island,
            CountryAlpha2::British_Indian_Ocean_Territory,
            CountryAlpha2::Brunei_Darussalam,
            CountryAlpha2::Bulgaria,
            CountryAlpha2::Burkina_Faso,
            CountryAlpha2::Burundi,
            CountryAlpha2::Cabo_Verde,
            CountryAlpha2::Cameroon,
            CountryAlpha2::Central_African_Republic,
            CountryAlpha2::Chad,
            CountryAlpha2::China,
            CountryAlpha2::Christmas_Island,
            CountryAlpha2::Cocos_Islands,
            CountryAlpha2::Comoros,
            CountryAlpha2::Congo,
            CountryAlpha2::Congo_Democratic_Republic,
            CountryAlpha2::Cook_Islands,
            CountryAlpha2::Costa_Rica,
            CountryAlpha2::Cote_d_Ivoire,
            CountryAlpha2::Croatia,
            CountryAlpha2::Cuba,
            CountryAlpha2::Curacao,
            CountryAlpha2::Cyprus,
            CountryAlpha2::Czechia,
            CountryAlpha2::Denmark,
            CountryAlpha2::Djibouti,
            CountryAlpha2::Ecuador,
            CountryAlpha2::Egypt,
            CountryAlpha2::El_Salvador,
            CountryAlpha2::Equatorial_Guinea,
            CountryAlpha2::Eritrea,
            CountryAlpha2::Estonia,
            CountryAlpha2::Eswatini,
            CountryAlpha2::Ethiopia,
            CountryAlpha2::Falkland_Islands,
            CountryAlpha2::Faroe_Islands,
            CountryAlpha2::Fiji,
            CountryAlpha2::Finland,
            CountryAlpha2::France,
            CountryAlpha2::French_Guiana,
            CountryAlpha2::French_Polynesia,
            CountryAlpha2::French_Southern_Territories,
            CountryAlpha2::Gabon,
            CountryAlpha2::Gambia,
            CountryAlpha2::Georgia,
            CountryAlpha2::Germany,
            CountryAlpha2::Ghana,
            CountryAlpha2::Gibraltar,
            CountryAlpha2::Greece,
            CountryAlpha2::Greenland,
            CountryAlpha2::Guadeloupe,
            CountryAlpha2::Guatemala,
            CountryAlpha2::Guernsey,
            CountryAlpha2::Guinea,
            CountryAlpha2::Guinea_Bissau,
            CountryAlpha2::Haiti,
            CountryAlpha2::Heard_Island_and_McDonald_Islands,
            CountryAlpha2::Holy_See,
            CountryAlpha2::Honduras,
            CountryAlpha2::Hungary,
            CountryAlpha2::Iceland,
            CountryAlpha2::India,
            CountryAlpha2::Iran,
            CountryAlpha2::Iraq,
            CountryAlpha2::Ireland,
            CountryAlpha2::Isle_of_Man,
            CountryAlpha2::Israel,
            CountryAlpha2::Italy,
            CountryAlpha2::Jersey,
            CountryAlpha2::Jordan,
            CountryAlpha2::Kiribati,
            CountryAlpha2::Korea_Democratic_Peoples_Republic,
            CountryAlpha2::Kuwait,
            CountryAlpha2::Lao_Peoples_Democratic_Republic,
            CountryAlpha2::Latvia,
            CountryAlpha2::Lebanon,
            CountryAlpha2::Lesotho,
            CountryAlpha2::Liberia,
            CountryAlpha2::Libya,
            CountryAlpha2::Liechtenstein,
            CountryAlpha2::Lithuania,
            CountryAlpha2::Luxembourg,
            CountryAlpha2::Macao,
            CountryAlpha2::Madagascar,
            CountryAlpha2::Malawi,
            CountryAlpha2::Malaysia,
            CountryAlpha2::Maldives,
            CountryAlpha2::Mali,
            CountryAlpha2::Malta,
            CountryAlpha2::Marshall_Islands,
            CountryAlpha2::Martinique,
            CountryAlpha2::Mauritania,
            CountryAlpha2::Mauritius,
            CountryAlpha2::Mayotte,
            CountryAlpha2::Mexico,
            CountryAlpha2::Moldova,
            CountryAlpha2::Monaco,
            CountryAlpha2::Montenegro,
            CountryAlpha2::Morocco,
            CountryAlpha2::Mozambique,
            CountryAlpha2::Myanmar,
            CountryAlpha2::Namibia,
            CountryAlpha2::Nauru,
            CountryAlpha2::Nepal,
            CountryAlpha2::Netherlands,
            CountryAlpha2::New_Caledonia,
            CountryAlpha2::New_Zealand,
            CountryAlpha2::Nicaragua,
            CountryAlpha2::Niger,
            CountryAlpha2::Niue,
            CountryAlpha2::Norfolk_Island,
            CountryAlpha2::North_Macedonia,
            CountryAlpha2::Norway,
            CountryAlpha2::Oman,
            CountryAlpha2::Pakistan,
            CountryAlpha2::Palestine,
            CountryAlpha2::Panama,
            CountryAlpha2::Papua_New_Guinea,
            CountryAlpha2::Paraguay,
            CountryAlpha2::Peru,
            CountryAlpha2::Philippines,
            CountryAlpha2::Pitcairn,
            CountryAlpha2::Poland,
            CountryAlpha2::Portugal,
            CountryAlpha2::Qatar,
            CountryAlpha2::Reunion,
            CountryAlpha2::Romania,
            CountryAlpha2::Rwanda,
            CountryAlpha2::Saint_Barthelemy,
            CountryAlpha2::Saint_Helena_Ascension_Tristan_da_Cunha,
            CountryAlpha2::Saint_Martin_French_part,
            CountryAlpha2::Saint_Pierre_and_Miquelon,
            CountryAlpha2::San_Marino,
            CountryAlpha2::Sao_Tome_and_Principe,
            CountryAlpha2::Saudi_Arabia,
            CountryAlpha2::Senegal,
            CountryAlpha2::Serbia,
            CountryAlpha2::Seychelles,
            CountryAlpha2::Sierra_Leone,
            CountryAlpha2::Slovakia,
            CountryAlpha2::Slovenia,
            CountryAlpha2::Solomon_Islands,
            CountryAlpha2::Somalia,
            CountryAlpha2::South_Africa,
            CountryAlpha2::South_Georgia_South_Sandwich_Islands,
            CountryAlpha2::South_Sudan,
            CountryAlpha2::Spain,
            CountryAlpha2::Sri_Lanka,
            CountryAlpha2::Sudan,
            CountryAlpha2::Suriname,
            CountryAlpha2::Svalbard_Jan_Mayen,
            CountryAlpha2::Sweden,
            CountryAlpha2::Switzerland,
            CountryAlpha2::Syrian_Arab_Republic,
            CountryAlpha2::Timor_Leste,
            CountryAlpha2::Togo,
            CountryAlpha2::Tokelau,
            CountryAlpha2::Tonga,
            CountryAlpha2::Tunisia,
            CountryAlpha2::Turkey,
            CountryAlpha2::Tuvalu,
            CountryAlpha2::Ukraine,
            CountryAlpha2::United_Arab_Emirates,
            CountryAlpha2::United_Kingdom,
            CountryAlpha2::United_States_Outlying_Islands,
            CountryAlpha2::Uruguay,
            CountryAlpha2::Uzbekistan,
            CountryAlpha2::Vanuatu,
            CountryAlpha2::Venezuela,
            CountryAlpha2::Viet_Nam,
            CountryAlpha2::Wallis_and_Futuna,
            CountryAlpha2::Western_Sahara,
            CountryAlpha2::Yemen,
            CountryAlpha2::Zambia,
            CountryAlpha2::Zimbabwe => self::_00,
            CountryAlpha2::Kenya,
            CountryAlpha2::Singapore,
            CountryAlpha2::Tanzania,
            CountryAlpha2::Uganda => self::_000,
            CountryAlpha2::Guyana,
            CountryAlpha2::Hong_Kong,
            CountryAlpha2::Mongolia => self::_001,
            CountryAlpha2::Australia => self::_0011,
            CountryAlpha2::Nigeria => self::_009,
            CountryAlpha2::Brazil,
            CountryAlpha2::Cambodia,
            CountryAlpha2::Colombia,
            CountryAlpha2::Indonesia,
            CountryAlpha2::Korea_Republic,
            CountryAlpha2::Taiwan_Province_of_China,
            CountryAlpha2::Thailand => self::_00x,
            CountryAlpha2::Japan => self::_010,
            CountryAlpha2::American_Samoa,
            CountryAlpha2::Anguilla,
            CountryAlpha2::Antigua_and_Barbuda,
            CountryAlpha2::Bahamas,
            CountryAlpha2::Barbados,
            CountryAlpha2::Bermuda,
            CountryAlpha2::Virgin_Islands_British,
            CountryAlpha2::Canada,
            CountryAlpha2::Cayman_Islands,
            CountryAlpha2::Dominica,
            CountryAlpha2::Dominican_Republic,
            CountryAlpha2::Grenada,
            CountryAlpha2::Guam,
            CountryAlpha2::Jamaica,
            CountryAlpha2::Micronesia,
            CountryAlpha2::Montserrat,
            CountryAlpha2::Northern_Mariana_Islands,
            CountryAlpha2::Palau,
            CountryAlpha2::Puerto_Rico,
            CountryAlpha2::Saint_Kitts_and_Nevis,
            CountryAlpha2::Saint_Lucia,
            CountryAlpha2::Saint_Vincent_and_the_Grenadines,
            CountryAlpha2::Sint_Maarten_Dutch_part,
            CountryAlpha2::Trinidad_and_Tobago,
            CountryAlpha2::Turks_and_Caicos_Islands,
            CountryAlpha2::United_States_of_America,
            CountryAlpha2::Virgin_Islands_U_S => self::_011,
            CountryAlpha2::Chile => self::_1xx0,
            CountryAlpha2::Belarus,
            CountryAlpha2::Kazakhstan,
            CountryAlpha2::Kyrgyzstan,
            CountryAlpha2::Russian_Federation,
            CountryAlpha2::Tajikistan,
            CountryAlpha2::Turkmenistan => self::_8_10,
        };
    }

    /** @return list<CountryAlpha2> */
    public function getCountriesAlpha2(): array {
        return match($this) {
            self::_0 => [
                CountryAlpha2::Samoa,
            ],
            self::_00 => [
                CountryAlpha2::Afghanistan,
                CountryAlpha2::Aland_Islands,
                CountryAlpha2::Albania,
                CountryAlpha2::Algeria,
                CountryAlpha2::Andorra,
                CountryAlpha2::Angola,
                CountryAlpha2::Antarctica,
                CountryAlpha2::Argentina,
                CountryAlpha2::Armenia,
                CountryAlpha2::Aruba,
                CountryAlpha2::Austria,
                CountryAlpha2::Azerbaijan,
                CountryAlpha2::Bahrain,
                CountryAlpha2::Bangladesh,
                CountryAlpha2::Belgium,
                CountryAlpha2::Belize,
                CountryAlpha2::Benin,
                CountryAlpha2::Bhutan,
                CountryAlpha2::Bolivia,
                CountryAlpha2::Bonaire_Sint_Eustatius_and_Saba,
                CountryAlpha2::Bosnia_and_Herzegovina,
                CountryAlpha2::Botswana,
                CountryAlpha2::Bouvet_Island,
                CountryAlpha2::British_Indian_Ocean_Territory,
                CountryAlpha2::Brunei_Darussalam,
                CountryAlpha2::Bulgaria,
                CountryAlpha2::Burkina_Faso,
                CountryAlpha2::Burundi,
                CountryAlpha2::Cabo_Verde,
                CountryAlpha2::Cameroon,
                CountryAlpha2::Central_African_Republic,
                CountryAlpha2::Chad,
                CountryAlpha2::China,
                CountryAlpha2::Christmas_Island,
                CountryAlpha2::Cocos_Islands,
                CountryAlpha2::Comoros,
                CountryAlpha2::Congo,
                CountryAlpha2::Congo_Democratic_Republic,
                CountryAlpha2::Cook_Islands,
                CountryAlpha2::Costa_Rica,
                CountryAlpha2::Cote_d_Ivoire,
                CountryAlpha2::Croatia,
                CountryAlpha2::Cuba,
                CountryAlpha2::Curacao,
                CountryAlpha2::Cyprus,
                CountryAlpha2::Czechia,
                CountryAlpha2::Denmark,
                CountryAlpha2::Djibouti,
                CountryAlpha2::Ecuador,
                CountryAlpha2::Egypt,
                CountryAlpha2::El_Salvador,
                CountryAlpha2::Equatorial_Guinea,
                CountryAlpha2::Eritrea,
                CountryAlpha2::Estonia,
                CountryAlpha2::Eswatini,
                CountryAlpha2::Ethiopia,
                CountryAlpha2::Falkland_Islands,
                CountryAlpha2::Faroe_Islands,
                CountryAlpha2::Fiji,
                CountryAlpha2::Finland,
                CountryAlpha2::France,
                CountryAlpha2::French_Guiana,
                CountryAlpha2::French_Polynesia,
                CountryAlpha2::French_Southern_Territories,
                CountryAlpha2::Gabon,
                CountryAlpha2::Gambia,
                CountryAlpha2::Georgia,
                CountryAlpha2::Germany,
                CountryAlpha2::Ghana,
                CountryAlpha2::Gibraltar,
                CountryAlpha2::Greece,
                CountryAlpha2::Greenland,
                CountryAlpha2::Guadeloupe,
                CountryAlpha2::Guatemala,
                CountryAlpha2::Guernsey,
                CountryAlpha2::Guinea,
                CountryAlpha2::Guinea_Bissau,
                CountryAlpha2::Haiti,
                CountryAlpha2::Heard_Island_and_McDonald_Islands,
                CountryAlpha2::Holy_See,
                CountryAlpha2::Honduras,
                CountryAlpha2::Hungary,
                CountryAlpha2::Iceland,
                CountryAlpha2::India,
                CountryAlpha2::Iran,
                CountryAlpha2::Iraq,
                CountryAlpha2::Ireland,
                CountryAlpha2::Isle_of_Man,
                CountryAlpha2::Israel,
                CountryAlpha2::Italy,
                CountryAlpha2::Jersey,
                CountryAlpha2::Jordan,
                CountryAlpha2::Kiribati,
                CountryAlpha2::Korea_Democratic_Peoples_Republic,
                CountryAlpha2::Kuwait,
                CountryAlpha2::Lao_Peoples_Democratic_Republic,
                CountryAlpha2::Latvia,
                CountryAlpha2::Lebanon,
                CountryAlpha2::Lesotho,
                CountryAlpha2::Liberia,
                CountryAlpha2::Libya,
                CountryAlpha2::Liechtenstein,
                CountryAlpha2::Lithuania,
                CountryAlpha2::Luxembourg,
                CountryAlpha2::Macao,
                CountryAlpha2::Madagascar,
                CountryAlpha2::Malawi,
                CountryAlpha2::Malaysia,
                CountryAlpha2::Maldives,
                CountryAlpha2::Mali,
                CountryAlpha2::Malta,
                CountryAlpha2::Marshall_Islands,
                CountryAlpha2::Martinique,
                CountryAlpha2::Mauritania,
                CountryAlpha2::Mauritius,
                CountryAlpha2::Mayotte,
                CountryAlpha2::Mexico,
                CountryAlpha2::Moldova,
                CountryAlpha2::Monaco,
                CountryAlpha2::Montenegro,
                CountryAlpha2::Morocco,
                CountryAlpha2::Mozambique,
                CountryAlpha2::Myanmar,
                CountryAlpha2::Namibia,
                CountryAlpha2::Nauru,
                CountryAlpha2::Nepal,
                CountryAlpha2::Netherlands,
                CountryAlpha2::New_Caledonia,
                CountryAlpha2::New_Zealand,
                CountryAlpha2::Nicaragua,
                CountryAlpha2::Niger,
                CountryAlpha2::Niue,
                CountryAlpha2::Norfolk_Island,
                CountryAlpha2::North_Macedonia,
                CountryAlpha2::Norway,
                CountryAlpha2::Oman,
                CountryAlpha2::Pakistan,
                CountryAlpha2::Palestine,
                CountryAlpha2::Panama,
                CountryAlpha2::Papua_New_Guinea,
                CountryAlpha2::Paraguay,
                CountryAlpha2::Peru,
                CountryAlpha2::Philippines,
                CountryAlpha2::Pitcairn,
                CountryAlpha2::Poland,
                CountryAlpha2::Portugal,
                CountryAlpha2::Qatar,
                CountryAlpha2::Reunion,
                CountryAlpha2::Romania,
                CountryAlpha2::Rwanda,
                CountryAlpha2::Saint_Barthelemy,
                CountryAlpha2::Saint_Helena_Ascension_Tristan_da_Cunha,
                CountryAlpha2::Saint_Martin_French_part,
                CountryAlpha2::Saint_Pierre_and_Miquelon,
                CountryAlpha2::San_Marino,
                CountryAlpha2::Sao_Tome_and_Principe,
                CountryAlpha2::Saudi_Arabia,
                CountryAlpha2::Senegal,
                CountryAlpha2::Serbia,
                CountryAlpha2::Seychelles,
                CountryAlpha2::Sierra_Leone,
                CountryAlpha2::Slovakia,
                CountryAlpha2::Slovenia,
                CountryAlpha2::Solomon_Islands,
                CountryAlpha2::Somalia,
                CountryAlpha2::South_Africa,
                CountryAlpha2::South_Georgia_South_Sandwich_Islands,
                CountryAlpha2::South_Sudan,
                CountryAlpha2::Spain,
                CountryAlpha2::Sri_Lanka,
                CountryAlpha2::Sudan,
                CountryAlpha2::Suriname,
                CountryAlpha2::Svalbard_Jan_Mayen,
                CountryAlpha2::Sweden,
                CountryAlpha2::Switzerland,
                CountryAlpha2::Syrian_Arab_Republic,
                CountryAlpha2::Timor_Leste,
                CountryAlpha2::Togo,
                CountryAlpha2::Tokelau,
                CountryAlpha2::Tonga,
                CountryAlpha2::Tunisia,
                CountryAlpha2::Turkey,
                CountryAlpha2::Tuvalu,
                CountryAlpha2::Ukraine,
                CountryAlpha2::United_Arab_Emirates,
                CountryAlpha2::United_Kingdom,
                CountryAlpha2::United_States_Outlying_Islands,
                CountryAlpha2::Uruguay,
                CountryAlpha2::Uzbekistan,
                CountryAlpha2::Vanuatu,
                CountryAlpha2::Venezuela,
                CountryAlpha2::Viet_Nam,
                CountryAlpha2::Wallis_and_Futuna,
                CountryAlpha2::Western_Sahara,
                CountryAlpha2::Yemen,
                CountryAlpha2::Zambia,
                CountryAlpha2::Zimbabwe,
            ],
            self::_000 => [
                CountryAlpha2::Kenya,
                CountryAlpha2::Singapore,
                CountryAlpha2::Tanzania,
                CountryAlpha2::Uganda,
            ],
            self::_001 => [
                CountryAlpha2::Guyana,
                CountryAlpha2::Hong_Kong,
                CountryAlpha2::Mongolia,
            ],
            self::_0011 => [
                CountryAlpha2::Australia
            ],
            self::_009 => [
                CountryAlpha2::Nigeria,
            ],
            self::_00x => [
                CountryAlpha2::Brazil,
                CountryAlpha2::Cambodia,
                CountryAlpha2::Colombia,
                CountryAlpha2::Indonesia,
                CountryAlpha2::Korea_Republic,
                CountryAlpha2::Taiwan_Province_of_China,
                CountryAlpha2::Thailand,
            ],
            self::_010 => [
                CountryAlpha2::Japan,
            ],
            self::_011 => [
                CountryAlpha2::American_Samoa,
                CountryAlpha2::Anguilla,
                CountryAlpha2::Antigua_and_Barbuda,
                CountryAlpha2::Bahamas,
                CountryAlpha2::Barbados,
                CountryAlpha2::Bermuda,
                CountryAlpha2::Virgin_Islands_British,
                CountryAlpha2::Canada,
                CountryAlpha2::Cayman_Islands,
                CountryAlpha2::Dominica,
                CountryAlpha2::Dominican_Republic,
                CountryAlpha2::Grenada,
                CountryAlpha2::Guam,
                CountryAlpha2::Jamaica,
                CountryAlpha2::Micronesia,
                CountryAlpha2::Montserrat,
                CountryAlpha2::Northern_Mariana_Islands,
                CountryAlpha2::Palau,
                CountryAlpha2::Puerto_Rico,
                CountryAlpha2::Saint_Kitts_and_Nevis,
                CountryAlpha2::Saint_Lucia,
                CountryAlpha2::Saint_Vincent_and_the_Grenadines,
                CountryAlpha2::Sint_Maarten_Dutch_part,
                CountryAlpha2::Trinidad_and_Tobago,
                CountryAlpha2::Turks_and_Caicos_Islands,
                CountryAlpha2::United_States_of_America,
                CountryAlpha2::Virgin_Islands_U_S,
            ],
            self::_1xx0 => [
                CountryAlpha2::Chile,
            ],
            self::_8_10 => [
                CountryAlpha2::Belarus,
                CountryAlpha2::Kazakhstan,
                CountryAlpha2::Kyrgyzstan,
                CountryAlpha2::Russian_Federation,
                CountryAlpha2::Tajikistan,
                CountryAlpha2::Turkmenistan,
            ],
            self::_119 => [],
        };
    }
}
